"""
springmass.py
"""
import numpy as np
from scipy.integrate import odeint

# define the rhs function, f
def f(u,t,k,m):
    return [u[1],-(k/m)*u[0]]

# define the rhs function, fnonlinear
def f_nonlinear(u,t,k,m,beta):
    return [u[1],-(k/m)*(u[0] + beta*(u[0]**3))]

T = 10.0    # final time until which we compute
N = 100     # number of time steps
taxis = np.linspace(0,T,N+1)
k = 5.0
beta = 0.1
m = 1.0

# compute frequency alpha for linear (Hookean) solution
alpha = np.sqrt(k/m)

# initial values for position and velocity
x0 = 1.0
v0 = 0.0
u0 = [x0,v0]

# compute parameter A and B of exact linear pendulum
B = x0
A = v0/alpha

# compute exact solution for linear pendulum
xexact = A*np.sin(alpha*taxis) + B*np.cos(alpha*taxis)

# use odeint to solve
t = np.linspace(0,T,N+1)
u = odeint(f,u0,t,args=(k,m,))
u_nonlinear = odeint(f_nonlinear,u0,t,args=(k,m,beta,))

# plot out results
import matplotlib.pyplot as plt
plt.plot(t,u[:,0],'bo')
plt.plot(t,xexact,'k-')
plt.plot(t,u_nonlinear[:,0],'rx-')
plt.xlim([0,10])
plt.ylim([-1.1,1.1])
plt.xlabel('Time')
plt.ylabel(r'$\theta$')
plt.legend(['odeint','Exact linear','odeint nonlinear'])
plt.savefig('springmass.jpg')