"""
carbon_euler.py
"""
import numpy as np
T = 0.5    # final time until which we compute
N = 10     # number of time steps
taxis = np.linspace(0,T,N+1)
dt = T/N    # length of each time step
lam = 2.0    # decay constant
r0 = 1.0    # ratio at t=0 to 1.0

# preallocate two arrays to store all values computed with explicit and
# implicit Euler - this will save a bit of time compared to appending a
# value in each step. The 1 is because the number of components in case of
# the decay equations is one.
rexp = np.zeros(N+1)
rimp = np.zeros(N+1)

# first entry is the initial value r0 for both
rexp[0] = r0
rimp[0] = r0

# forward euler
for i in range(N):
    rexp[i+1] = rexp[i] -dt*lam*rexp[i]
    
# backward euler
for i in range(N):
    rimp[i+1] = rimp[i]/(1 + dt*lam)

# plot out results
import matplotlib.pyplot as plt
plot1 = plt.figure(1)
plt.plot(taxis,rexp,'ro')
plt.plot(taxis,rimp,'bo')
plt.plot(taxis,r0*np.exp(-lam*taxis),'k-')
plt.xlim([0,0.5])
plt.ylim([0.3,1.0])
plt.xlabel('Time')
plt.ylabel('C14')
plt.legend(['Explicit Euler','Implicit Euler','Exact solution'])
plt.savefig('carbon_euler.jpg')

# to compute the error, we have to figure out the exact solutions at all
# time points - note that taxis is a row vector whereas rexp, rimp are
# column vectors, so we need to transpose taxis to match
rexact = np.transpose(r0*np.exp(-lam*taxis))

# now compute error of forward and backward Euler at each step
errorexp = np.abs(rexp-rexact)
errorimp = np.abs(rimp-rexact)
plot2 = plt.figure(2)
plt.semilogy(taxis,errorexp,'r')
plt.semilogy(taxis,errorimp,'b')
plt.xlim([0.05,0.5])
plt.xlabel('Time')
plt.ylabel('Error')
plt.legend(['Explicit Euler','Implicit Euler'])
plt.savefig('carbon_euler2.jpg')