"""
springmass_linear_euler.py
Solves the equation for the linear spring-mass system with forward and
backward Euler methods
"""
import numpy as np
from scipy.integrate import odeint

# define the rhs function for linear spring mass case
def f(u,t,A):
    return np.matmul(A,u)

T = 10.0    # final time until which we compute
N = 200     # number of time steps
taxis = np.linspace(0,T,N+1)
dt = T/N
k = 5.0
b = 0.0
beta = 0.1
m = 1.0

# define the matrix A
A = np.array([[0, 1], [-k, -b]])

# initial values for position and velocity
x0 = 1.0
v0 = 0.0
u0 = [x0,v0]

# allocate vectors to store solution; note that for the pendulum the vector u
# has two components
u_exp = np.zeros([N+1,2])
u_exp[0,:] = u0

# define right hand side function; assume u = [ x, v ] so that u[0]=x,
# u[1]=v. % Note that we allow for an argument t that we do not really need, so that
# we can reuse f later for the odeint function.

# simple Forward Euler first
for i in range(N):
    u_exp[i+1,:] = u_exp[i,:] + dt*f(u_exp[i,:],(i+1)*dt,A)

# now Backward Euler
u_imp = np.zeros([N+1,2])
u_imp[0,:] = u0

M = np.identity(2) - dt*A
for i in range(N):
    u_imp[i+1,:] = np.linalg.solve(M,u_imp[i,:])

# solve with odeint
u_odeint = odeint(f,u0,taxis,args=(A,))

# plot out results
import matplotlib.pyplot as plt
plt.plot(taxis,u_exp[:,0],'r')
plt.plot(taxis,u_imp[:,0],'b')
plt.plot(taxis,u_odeint[:,0],'k-')
plt.xlim([0,10])
plt.ylim([-2.5,2.5])
plt.xlabel('Time')
plt.ylabel('x')
plt.legend(['Explicit','Implicit','odeint'])
plt.savefig('springmass_linear_euler.jpg')