"""
carbon_euler_workprecision.py
Solves the decay equation with forward and backward Euler for a range of
time steps to analyse how the error decreases as we make dt smaller
"""
import numpy as np
  
############################################################################
# Functions
############################################################################ 
# exact solution
def u_exact(t,r0,lam):
    return r0*np.exp(-lam*t)

# forward euler function
def exp_euler(u0,Tend,nsteps,lam):
    dt = Tend/nsteps
    u = np.zeros(nsteps+1)
    u[0] = u0
    for i in range(nsteps):
        u[i+1] = u[i] - dt*lam*u[i]  
    return u

# backward euler function
def imp_euler(u0,Tend,nsteps,lam):    
    dt = Tend/nsteps
    u = np.zeros(nsteps+1)
    u[0] = u0
    for i in range(nsteps):
        u[i+1] = u[i]/(1 + dt*lam)  
    return u

# function definition of time derivative
def f(u,lam):
    return -lam*u

# heun's method
def heun(u0,Tend,nsteps,lam):    
    dt = Tend/nsteps
    u = np.zeros(nsteps+1)
    u[0] = u0
    for i in range(nsteps):
        utemp = u[i] + dt*f(u[i],lam)
        u[i+1] = u[i] + 0.5*dt*(f(u[i],lam) + f(utemp,lam))
        
    return u

############################################################################
    
# set up problem parameters
T = 1.0   # time up to which we compute
lam = 1.0   # decay constant
r0 = 1.0   # set ratio at t=0
N = [1000,750,500,250,100,75,50,10]

# allocate vectors to store for every run
err_exp = np.zeros(len(N))
err_heun = np.zeros(len(N))

# Instead of dt, we now store the workload, measured in the number of times
# that a method has to evaluate the right hand side function
workload_exp = np.zeros(len(N))
workload_heun = np.zeros(len(N))

for n in range(len(N)):
    taxis = np.linspace(0,T,N[n]+1)   # add +1 to account for t=0
    u_exp = exp_euler(r0,T,N[n],lam)
    u_heun = heun(r0,T,N[n],lam)

    # stor the time step dt for plotting
    workload_exp[n] = N[n]
    workload_heun[n] = 2*N[n]
    
    # now compute the errors
    err_exp[n] = max(np.abs(u_exp-u_exact(taxis,r0,lam)))
    err_heun[n] = max(np.abs(u_heun-u_exact(taxis,r0,lam)))

# Find slope of lines for Euler and Heun's method
p_exp = np.polyfit(np.log(workload_exp), np.log(err_exp),1)
p_heun = np.polyfit(np.log(workload_heun), np.log(err_heun),1)

# plot out results
import matplotlib.pyplot as plt
plot1 = plt.figure(1)
plt.loglog(workload_exp,err_exp,'ro-')
plt.loglog(workload_heun,err_heun,'go-')
txt_exp='Slope p='+str(round(p_exp[0],2))
plt.text(7e2,1e-3,txt_exp)
txt_heun='Slope p='+str(round(p_heun[0],2))
plt.text(7e2,1e-6,txt_heun)
plt.xlabel('Workload')
plt.ylabel('Error')
plt.legend(['Explicit Euler','Heun\'s method'])
plt.savefig('carbon_euler_precision.jpg')
