# Program to calculate homogeneous chemistry based on theory developed by Nesic
# and co-workers. Theory being used from following references:
# Nordsveen et al. Corrosion 59(5), 443-456.

import numpy as np
import matplotlib.pyplot as plt
from scipy.integrate import odeint
from chemical_functions import set_chemistry, dxdt_5_pHvariable

# define chemistry data dictionary
chemparam = set_chemistry()
cFe = 0.0   # Fe2+ concentration = 0 in bulk solution

# define arrays for solution of chemical equations
ndim = 5  # number of independent chemical reactions

# create constants for use in the implicit solution scheme
c1 = chemparam['c1']
c2 = chemparam['c2']
c3 = chemparam['c3']
c4 = chemparam['c4']
c5 = chemparam['c5']
c6 = chemparam['c6']
c7 = chemparam['c7']
c8 = chemparam['c8']
c9 = chemparam['c9']
c10 = chemparam['c10']

def f(y, t):
    
        x1 = y[0]   # concentration of cO2 (mol/m3)
        x2 = y[1]   # concentration of H2CO3 (mol/m3)
        x3 = y[2]   # concentration of HCO3- (mol/m3)
        x4 = y[3]   # concentration of CO32- (mol/m3)
        x5 = y[4]   # concentration of OH- (mol/m3)
        
        [f0,f1,f2,f3,f4]= dxdt_5_pHvariable(x1,x2,x3,x4,x5,chemparam,0.0)
        return [f0, f1, f2, f3, f4]

# initialise the concentrations CO2, H2CO3, HCO3-, CO32- and COH-. in practice the 
# solutions are not found to be sensitive to these specific values
xb0 = 10**-4             
cCO2_init = chemparam['cCO2_init']                
y0 = [cCO2_init, xb0, xb0, xb0, xb0]       

# t  = np.linspace(0,1)   # time grid
t  = np.linspace(0,1.0,101)   # time grid

# solve the time dependent chemical equations using PYTHON functions
soln = odeint(f, y0, t)
cCO2 = soln[:, 0]   # concentration of CO2 at ith time step
cH2CO3 = soln[:, 1]   # concentration of H2CO3 at ith time step
cHCO3 = soln[:, 2]   # concentration of HCO3- at ith time step
cCO3 = soln[:, 3]   # concentration of CO32- at ith time step
cOH = soln[:, 4]   # concentration of OH-- at ith time step
# calculate concentration of H+ by  charge balance equation
cH = np.zeros(len(cOH),dtype=float)
for j in range(len(cOH)):
    cH[j] = cHCO3[j] + 2*cCO3[j] + cOH[j]

# set steady state concentrations in bulk
cCO2_steady = cCO2[-1]
cH2CO3_steady = cH2CO3[-1]
cHCO3_steady = cHCO3[-1]
cCO3_steady = cCO3[-1]
cOH_steady = cOH[-1]
cH_steady = cH[-1]
pH_steady = -np.log10(cH_steady/1000)
print("cCO2 steady = {0:10.5e}".format(cCO2_steady))
print("cH2CO3 steady = {0:10.5e}".format(cH2CO3_steady))
print("CHCO3 steady = {0:10.5e}".format(cHCO3_steady))
print("cCO3 steady = {0:10.5e}".format(cCO3_steady))
print("cOH steady = {0:10.5e}".format(cOH_steady))
print("cH steady = {0:10.5e}".format(cH_steady))
print("pH steady = {0:10.5e}".format(pH_steady))

# plot out chemical concentrations
plt.ion()
fig=plt.figure()
plt.semilogy(t,cCO2,'k-',markersize=5,label='CO2')         
plt.semilogy(t,cH2CO3,'b-',markersize=5,label='H2CO3')         
plt.semilogy(t,cHCO3,'r-',markersize=5,label='HC03-')         
plt.semilogy(t,cCO3,'g-',markersize=5,label='C032-')         
plt.semilogy(t,cOH,'m-',markersize=5,label='OH-')
plt.semilogy(t,cH,'y-',markersize=5,label='H+')

plt.legend(loc='best')
plt.xlabel('time (secs)',style='italic')
plt.ylabel('Species concentrations (mol/m3)',style='italic')

