"""
carbon_euler_accuracy.py
Solves the decay equation with forward and backward Euler for a range of
time steps to analyse how the error decreases as we make dt smaller
"""
import numpy as np
    
# exact solution
def u_exact(t,r0,lam):
    return r0*np.exp(-lam*t)

# forward euler function
def exp_euler(u0,Tend,nsteps,lam):
    dt = Tend/nsteps
    u = np.zeros(nsteps+1)
    u[0] = u0
    for i in range(nsteps):
        u[i+1] = u[i] - dt*lam*u[i]  
    return u

# backward euler function
def imp_euler(u0,Tend,nsteps,lam):    
    dt = Tend/nsteps
    u = np.zeros(nsteps+1)
    u[0] = u0
    for i in range(nsteps):
        u[i+1] = u[i]/(1 + dt*lam)  
    return u
        
# set up problem parameters
T = 1.0   # time up to which we compute
lam = 1.0   # decay constant
r0 = 1.0   # set ratio at t=0
N = [1000,750,500,250,100,75,50,10]

# allocate vectors to store for every run
err_exp = np.zeros(len(N))
err_imp = np.zeros(len(N))
dts = np.zeros(len(N))

# exact solution
# taxis = np.linspace(0,T,N[0])
# out = u_exact(taxis,r0,lam)

for n in range(len(N)):
    taxis = np.linspace(0,T,N[n]+1)
    u_exp = exp_euler(r0,T,N[n],lam)
    u_imp = imp_euler(r0,T,N[n],lam)

    # stor the time step dt for plotting
    dts[n] = taxis[1] - taxis[0]
    
    # now compute the errors
    err_exp[n] = max(np.abs(u_exp-u_exact(taxis,r0,lam)))
    err_imp[n] = max(np.abs(u_imp-u_exact(taxis,r0,lam)))

# we fit a line log(err) = p*log(N) + C through the data points for
# reasons that will become clear later
# build 6th order fit to build data
p_exp = np.polyfit(np.log(dts), np.log(err_exp),1)
p_imp = np.polyfit(np.log(dts), np.log(err_imp),1)

# plot out results
import matplotlib.pyplot as plt
plot1 = plt.figure(1)
plt.loglog(dts,err_exp,'ro')
plt.loglog(dts,np.exp(np.polyval(p_exp, np.log(dts))),'r')
plt.xlim([dts[0],dts[len(N)-1]])
figtext='Slope p='+str(round(p_exp[0],2))
plt.text(1e-2,1e-3,figtext)
plt.xlabel(r'$\Delta t$')
plt.ylabel('Error')
plt.legend(['Explicit Eulwer','Linear Fit'])
plt.savefig('carbon_euler_accuracy1.jpg')

plot1 = plt.figure(2)
plt.loglog(dts,err_imp,'bo')
plt.loglog(dts,np.exp(np.polyval(p_imp, np.log(dts))),'b',size=2)
plt.xlim([dts[0],dts[len(N)-1]])
figtext='Slope p='+str(round(p_imp[0],2))
plt.text(1e-2,1e-3,figtext)
plt.xlabel(r'$\Delta t$')
plt.ylabel('Error')
plt.legend(['Implicit Eulwer','Linear Fit'])
plt.savefig('carbon_euler_accuracy2.jpg')

