"""
 FIV_natural_frequency_fdm.py
 Calculates the critical velocity for instability of a
 FIV system with clamped-clamped ends 
"""
import numpy as np

##########################################################################
def natural_frequencies(E,L,I,mtot,mf,V,Nx):
    
    ##########################################################################
    # discretisation of pipe
    deltax = L/(Nx-1)    # spatial grid spacing

    ##########################################################################
    # defining the constants for the set of linear finite difference euqations
    a = (E*I)/(mtot*(deltax)**4)
    b = (mf*(V**2))/(mtot*(deltax)**2)
    C1 = a
    C2 = b-4*a
    C3 = 6*a-2*b
    C4 = C2
    C5 = C1

    # creation of FDM matrix M
    n = Nx-2
    M = np.zeros([n,n])
    M[0][0]=C1+C3
    M[0][1]=C4
    M[0][2]=C5

    M[1][0]=C2
    M[1][1]=C3
    M[1][2]=C4
    M[1][3]=C5

    for k in range(2,n-2):
        M[k][k-2]=C1
        M[k][k-1]=C2
        M[k][k]=C3
        M[k][k+1]=C4
        M[k][k+2]=C5

    M[n-2][n-4]=C1
    M[n-2][n-3]=C2
    M[n-2][n-2]=C3
    M[n-2][n-1]=C4
    M[n-1][n-3]=C1
    M[n-1][n-2]=C2
    M[n-1][n-1]=C3+C5

    # calculate natural frequencies from the real parts of the first two eigenvalues of M
    evals, evecs = np.linalg.eig(M)
    omega_squared = evals[n-1]
    omega_natural = np.sqrt(evals)
    omega_1 = omega_natural[n-1].real
    omega_2 = omega_natural[n-2].real
    
    return omega_1,omega_2,omega_squared

# problem parameters from Dangal & Ghimire
steel_pipe = 1; aluminium_pipe = 2; CPVC_pipe = 3
material = 1
if (material == steel_pipe):
    # Steel pipe
    E = 207e9       # Young's modules of pipe (Pa)
    L = 3.048       # pipe clamp spacing (m)
    I = 8.73e-9     # moment of area of pipe (m^4)
    mtot = 1.386    # total mass of pipe and fluid per unit length (kg/m)
    mf = 0.38       # mass of fluid per unit length (kg/m)
elif (material == aluminium_pipe):
    # Aluminium pipe
    E = 68.9e9       # Young's modules of pipe (Pa)
    L = 3.048       # pipe clamp spacing (m)
    I = 8.73e-9     # moment of area of pipe (m^4)
    mtot = 0.715    # total mass of pipe and fluid per unit length (kg/m)
    mf = 0.38       # mass of fluid per unit length (kg/m)
elif (material == CPVC_pipe):
    # CPVC pipe
    E = 2.9e9       # Young's modules of pipe (Pa)
    L = 3.048       # pipe clamp spacing (m)
    I = 8.73e-9     # moment of area of pipe (m^4)
    mtot = 0.574    # total mass of pipe and fluid per unit length (kg/m)
    mf = 0.38       # mass of fluid per unit length (kg/m)
    
##########################################################################
# discretisation of pipe
Nx = 60

# allocate vectors to store errors for every run
Npts = 101
omega_squared = np.zeros(Npts)
V = np.linspace(50,150,Npts)

for n in range(Npts):
    om_1,om_2,om_squared = natural_frequencies(E,L,I,mtot,mf,V[n],Nx)
    omega_squared[n] = om_squared

# find velocity where square of natural frequency becomes negative
om1 = omega_squared[0]
crit_velocity = 0
for n in range(Npts-1):
    product = omega_squared[n]*omega_squared[n+1]
    if (product < 0):    # use linear interpolation for greater accuracy
        fac1 = np.abs(omega_squared[n])
        fac2 = np.abs(omega_squared[n+1])      
        crit_velocity = (fac2*V[n]+fac1*V[n+1])/(fac1+fac2)
        break

print('critical velocity = {0:6.3f}'.format(crit_velocity))

# plot out results
import matplotlib.pyplot as plt
plot1 = plt.figure(1)
plt.plot(V,omega_squared,'b-')
plt.xlabel('Fluid velocity')
plt.ylabel('Natural frequency^2 (rad/s)')
plt.title('Effect of velocity on stability of Flow Induced Vibration in a steel pipe')
plt.text(100,6000,'critical velocity={0:6.2f}'.format(crit_velocity))
plt.savefig('steel_velocity_stability_fdm.jpg')

