# functions used to set us equations for homoegeneous chemical reactions

import numpy as np

def dxdt_5_pHvariable(x1,x2,x3,x4,x5,chemparam,cFe):

    # set parameters from dictionary 
    c1 = chemparam['c1']
    c2 = chemparam['c2']
    c3 = chemparam['c3']
    c4 = chemparam['c4']
    c5 = chemparam['c5']
    c6 = chemparam['c6']
    c7 = chemparam['c7']
    c8 = chemparam['c8']
    c9 = chemparam['c9']
    c10 = chemparam['c10']   
    
    x6 = x3 + 2*x4 + x5 - 2*cFe  # cH
    # rate of change of CO2
    dx1dt = c1*x2 + c2*x1
    # rate of change of H2CO3
    dx2dt = -dx1dt -c3*x2 -c4*x3*x6
    # rate of change of HCO3-
    dx3dt = c3*x2 + c4*x3*x6 + c5*x3 + c6*x4*x6
    # rate of change of CO32-
    dx4dt = c7*x3+c8*x4*x6
    # rate of change of OH-
    dx5dt = c9 + c10*x5*x6
    
    return [dx1dt,dx2dt,dx3dt,dx4dt,dx5dt]

# function to set chemical parameters needed in solutions of chemical equations
def set_chemistry():
                                         
    # pH not specified but determined from chemical reaction equations
    # set ionic strength
    I=0.1711   # value for 1% NaCl 
    Tc=20
    pCO2=1   # 1 bar

    # total pressure P = PH2O + PCO2
    # PH2O calculated using steam tables at Tc. E.g. 25oC, PH2O = 0.0313 Atm
    P=14.8   # total pressure expressed in p.s.i.
    # PH=4     # specify pH and hence H+ concentration

    T=Tc+273.15   # Kelvin
    Tf=Tc*9/5 +32   # Fahrenheit
    Tk=Tc+273.15   # Kelvin

    # Calculate steady state chemistry in bulk solution
    #*****************************************************
    #CO2 aqueous. Ksol taken from Nordsveen et al (2003), Table 2, p 447.
    Ksol= (14.5/1.00258)*10**-(2.27+0.00565*Tf-8.06*(10**-6)*(Tf**2)
    +0.075*I)

    #dissociation of water. Reaction constants taken from Nordsveen et al (2003),
    # table 2, p 447.
    Kwa= 10**-(29.3868-0.0737549*Tk+7.47881*(10**-5)*Tk**2)
    Kbwa= 7.85* 10**10 *(Tc/Tc)
    Kfwa=Kwa*Kbwa

    #hydration of H2CO3. Reaction constants taken from Nordsveen et al (2003),
    # table 2, p 447.
    Khy=2.58*(10**-3) *(Tc/Tc)
    Kfhy=10**(329.85-110.541*np.log10(Tk)-(17265.4/Tk))
    Kbhy=Kfhy/Khy

    #dissociation of H2CO3. Reaction constants taken from Nordsveen et al (2003),
    # table 2, p 447.
    Kca=387.6*10**-(6.41-1.594*(10**(-3))*Tf
    +(8.52* 10**(-6) *Tf**(2))-3.07*10**(-5)*P-0.4772*(I**0.5)+0.118*I)
    Kfca= 10**(5.71+0.0526*Tc-2.94*10**-4 *Tc**2 +7.91*10**-7*Tc**3)
    Kbca= (Kfca/Kca)

    #dissociation of HCO3-(matched depends on variables). Reaction constants 
    # taken from Nordsveen et al (2003), table 2, p 447.
    Kbi = 10**-(10.61-4.97*(10**-3)*Tf+1.331*(10**-5)*(Tf**2)-2.624*(10**-5)*P-
    1.166*(I**0.5)+0.3466*I)
    Kfbi=10**9 *(Tc/Tc)
    Kbbi= Kfbi/Kbi 

    # calculate steady state concentrations of CO2, H2CO3 in mol/m3
    cCO2_init = pCO2*Ksol*1000
    cH2CO3_init = pCO2*Ksol*Khy*1000

    # create constants for use in the implicit solution scheme
    c1 = Kbhy
    c2 = -Kfhy
    c3 = Kfca
    c4 = -Kbca*0.001
    c5 = -Kfbi
    c6 = Kbbi*0.001
    c7 = Kfbi
    c8 = -Kbbi*0.001
    c9 = Kfwa*1000
    c10 = -Kbwa*0.001

    # set chemistry dictionary values
    chemparam = dict([('I',I),('Tc',Tc),('pcO2',pCO2),('P',P),('T',T),('Tf',Tf),
                      ('Tk',Tk),('cCO2_init',cCO2_init),('cH2CO3_init',cH2CO3_init),
                      ('c1',c1),('c2',c2),('c3',c3),('c4',c4),('c5',c5),('c6',c6),
                      ('c7',c7),('c8',c8),('c9',c9),('c10',c10)])
    return chemparam