"""
 FIV_natural_frequency_fdm.py
 Carries of grid convergence studies for the  natural frequencies in a 
 FIV system with simply supported-simply supported ends 
"""
import numpy as np

##########################################################################
def gridconv_natural_frequencies(E,L,I,mtot,mf,V,Nx):
    
    ##########################################################################
    # discretisation of pipe
    deltax = L/(Nx-1)    # spatial grid spacing

    ##########################################################################
    # defining the constants for the set of linear finite difference euqations
    a = (E*I)/(mtot*(deltax)**4)
    b = (mf*(V**2))/(mtot*(deltax)**2)
    C1 = a
    C2 = b-4*a
    C3 = 6*a-2*b
    C4 = C2
    C5 = C1

    # creation of FDM matrix M
    n = Nx-2
    M = np.zeros([n,n])
    M[0][0]=C3-C1
    M[0][1]=C4
    M[0][2]=C5

    M[1][0]=C2
    M[1][1]=C3
    M[1][2]=C4
    M[1][3]=C5

    for k in range(2,n-2):
        M[k][k-2]=C1
        M[k][k-1]=C2
        M[k][k]=C3
        M[k][k+1]=C4
        M[k][k+2]=C5

    M[n-2][n-4]=C1
    M[n-2][n-3]=C2
    M[n-2][n-2]=C3
    M[n-2][n-1]=C4
    M[n-1][n-3]=C1
    M[n-1][n-2]=C2
    M[n-1][n-1]=C3-C5

    # calculate natural frequencies from the real parts of the first two eigenvalues of M
    evals, evecs = np.linalg.eig(M)
    omega_natural = np.sqrt(evals)
    omega_1 = omega_natural[n-1].real
    omega_2 = omega_natural[n-2].real
    
    return omega_1,omega_2

# problem parameters from Dangal & Ghimire
steel_pipe = 1; aluminium_pipe = 2; CPVC_pipe = 3
material = 2
V = 13.1
if (material == steel_pipe):
    # Steel pipe
    E = 207e9       # Young's modules of pipe (Pa)
    L = 3.048       # pipe clamp spacing (m)
    I = 8.73e-9     # moment of area of pipe (m^4)
    mtot = 1.386    # total mass of pipe and fluid per unit length (kg/m)
    mf = 0.38       # mass of fluid per unit length (kg/m)
elif (material == aluminium_pipe):
    # Aluminium pipe
    E = 68.9e9       # Young's modules of pipe (Pa)
    L = 3.048       # pipe clamp spacing (m)
    I = 8.73e-9     # moment of area of pipe (m^4)
    mtot = 0.715    # total mass of pipe and fluid per unit length (kg/m)
    mf = 0.38       # mass of fluid per unit length (kg/m)
elif (material == CPVC_pipe):
    # CPVC pipe
    E = 2.9e9       # Young's modules of pipe (Pa)
    L = 3.048       # pipe clamp spacing (m)
    I = 8.73e-9     # moment of area of pipe (m^4)
    mtot = 0.574    # total mass of pipe and fluid per unit length (kg/m)
    mf = 0.38       # mass of fluid per unit length (kg/m)
    
##########################################################################
# discretisation of pipe
Nx = [10,20,30,40,50,60,70]

# allocate vectors to store errors for every run
omega_1 = np.zeros(len(Nx))

for n in range(len(Nx)):
    om_1,om_2 = gridconv_natural_frequencies(E,L,I,mtot,mf,V,Nx[n])
    omega_1[n] = om_1

print('Fine grid natural frequency= {0:6.3f}'.format(omega_1[len(Nx)-1]))

# plot out results
import matplotlib.pyplot as plt
plot1 = plt.figure(1)
plt.plot(Nx,omega_1,'b-o')
plt.xlabel('Number of nodes along the pipe')
plt.ylabel('Natural frequency (rad/s)')
plt.text(20,29.05,'fine grid frequency={0:6.2f}'.format(omega_1[len(Nx)-1]))
plt.title('Natural Frequencies of a simply-supported aluminium pipe, V=13.10m/s')
plt.savefig('al_gridconvergence_FIV_natural_frequency_fdm.jpg')

# Calculation of the order of accuracy of the method
Nxcoarse = 40; Nxmedium = 80; Nxfine = 160; r = 0.5;
omega_coarse,om_2 = gridconv_natural_frequencies(E,L,I,mtot,mf,V,Nxcoarse)
omega_medium,om_2 = gridconv_natural_frequencies(E,L,I,mtot,mf,V,Nxmedium)
omega_fine,om_2 = gridconv_natural_frequencies(E,L,I,mtot,mf,V,Nxfine)
p = np.log((omega_fine-omega_medium)/(omega_medium-omega_coarse))/np.log(r)
print('Estimate of order of convergence, p = {0:6.3f}'.format(p))