"""
springmass_euler.py
Solves the equation for the nonlinear spring-mass system with forward and
backward Euler methods
"""
import numpy as np
from scipy.integrate import odeint
from scipy.optimize import fsolve

# define the rhs function, fnonlinear
def f_nonlinear(u,t,k,m,beta):
    return [u[1],-(k/m)*(u[0] + beta*(u[0]**3))]

# define the function which needs to be solved at each implicit time step
def F(u,t,dt,i,u_init):
    return u - dt*np.array(f_nonlinear(u,(i+1)*dt,k,m,beta)) - u_init

T = 10.0    # final time until which we compute
N = 200     # number of time steps
taxis = np.linspace(0,T,N+1)
dt = T/N
k = 5.0
beta = 0.1
m = 1.0

# compute frequency alpha for linear (Hookean) solution
alpha = np.sqrt(k/m)

# initial values for position and velocity
x0 = 1.0
v0 = 0.0
u0 = [x0,v0]

# allocate vectors to store solution; note that for the pendulum the vector u
# has two components
u_exp = np.zeros([N+1,2])
u_exp[0,:] = u0

# define right hand side function; assume u = [ x, v ] so that u[0]=x,
# u[1]=v. % Note that we allow for an argument t that we do not really need, so that
# we can reuse f later for the odeint function.

# simple Forward Euler first
for i in range(N):
    dudt = np.array(f_nonlinear(u_exp[i,:],(i+1)*dt,k,m,beta))
    u_exp[i+1,:] = u_exp[i,:] + dt*dudt

# solve with odeint
u_nonlinear = odeint(f_nonlinear,u0,taxis,args=(k,m,beta,))

# now Backward Euler
u_imp = np.zeros([N+1,2])
u_imp[0,:] = u0
for i in range(N):
    u_init = u_imp[i,:]
    # val = fsolve(F,u_init,args=((i+1)*dt,dt,i,u_init))
    u_imp[i+1,:] = fsolve(F,u_init,args=((i+1)*dt,dt,i,u_init))
    
# plot out results
import matplotlib.pyplot as plt
plt.plot(taxis,u_exp[:,0],'r')
plt.plot(taxis,u_imp[:,0],'b')
plt.plot(taxis,u_nonlinear[:,0],'k-')
plt.xlim([0,10])
plt.ylim([-2.5,2.5])
plt.xlabel('Time')
plt.ylabel('x')
plt.legend(['Explicit','Implicit','odeint'])
plt.savefig('springmass_euler.jpg')